<?php

namespace App\Http\Controllers;

use App\Models\Blog;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class BlogController extends Controller
{
    /* =======================
        ADMIN SECTION
    ======================= */

    // Admin blog list
    public function index()
    {
        $blogs = Blog::latest()->get();
        return view('admin.blog.blog_index', compact('blogs'));
    }

    // Admin create form
    public function create()
    {
        return view('admin.blog.Addblog');
    }

    // Admin store blog
    public function store(Request $request)
    {
        $request->validate([
            'title'   => 'required|string|max:255',
            'content' => 'required|string',
            'image'   => 'nullable|image|mimes:jpg,jpeg,png|max:10240',
        ]);

        $imagePath = null;

        if ($request->hasFile('image')) {
            $image = $request->file('image');
            $filename = time().'_'.$image->getClientOriginalName();
            $image->move(public_path('blogs'), $filename);
            $imagePath = 'blogs/'.$filename;
        }

        Blog::create([
            'title'   => $request->title,
            'content' => $request->content,
            'image'   => $imagePath,
            'tags'    => $request->tags,
            'slug'    => Str::slug($request->title),
        ]);

        return redirect()->back()->with('success', 'Blog added successfully!');
    }

    // Admin edit form
    public function edit($id)
    {
        $blog = Blog::findOrFail($id);
        return view('admin.blog.edit', compact('blog'));
    }

    // Admin update blog
    public function update(Request $request, $id)
    {
        $blog = Blog::findOrFail($id);

        $request->validate([
            'title'   => 'required|string|max:255',
            'content' => 'required|string',
            'image'   => 'nullable|image|mimes:jpg,jpeg,png|max:10240',
        ]);

        if ($request->hasFile('image')) {

            // delete old image
            if ($blog->image && file_exists(public_path($blog->image))) {
                unlink(public_path($blog->image));
            }

            $image = $request->file('image');
            $filename = time().'_'.$image->getClientOriginalName();
            $image->move(public_path('blogs'), $filename);
            $blog->image = 'blogs/'.$filename;
        }

        $blog->title   = $request->title;
        $blog->content = $request->content;
        $blog->tags    = $request->tags;
        $blog->slug    = Str::slug($request->title);
        $blog->save();

        return redirect()->route('blog.view')->with('success', 'Blog updated successfully!');
    }

    // Admin delete blog
    public function destroy($id)
    {
        $blog = Blog::findOrFail($id);

        if ($blog->image && file_exists(public_path($blog->image))) {
            unlink(public_path($blog->image));
        }

        $blog->delete();

        return redirect()->route('blog.view')->with('success', 'Blog deleted successfully!');
    }

    /* =======================
        FRONTEND SECTION
    ======================= */

    // Frontend blog list (with pagination)
    public function blogList()
    {
        $blogs = Blog::latest()->paginate(6);
        return view('blog.index', compact('blogs'));
    }

    // Frontend single blog (by slug)
    public function show($slug)
    {
        $blog = Blog::where('slug', $slug)->firstOrFail();
        return view('blog.single-blog', compact('blog'));
    }
}
