<?php

namespace App\Http\Controllers;

use App\Models\ServiceOrder;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class ServiceOrderController extends Controller
{
    public function showOrderForm(Request $request)
    {
        $service = $request->input('service'); // seo, edu, niche
        $plan = $request->input('plan');
        $price = $request->input('price');
        $isPaid = $request->input('status') === 'paid';

        if (!$service || !$plan || !$price) {
            return redirect()->back()->with('error', 'Invalid plan selection.');
        }

        return view('pages.service-order-form', compact('service', 'plan', 'price', 'isPaid'));
    }

    public function processOrder(Request $request)
    {
        $request->validate([
            'service_type' => 'required|string',
            'plan_name' => 'required|string',
            'price' => 'required|numeric',
            'customer_email' => 'required|email',
            'target_url' => 'required|url',
            'topic' => 'nullable|string',
            'whatsapp_number' => 'nullable|string',
            'article_file' => 'nullable|file|mimes:doc,docx,pdf,txt|max:5120',
            'payment_status' => 'nullable|string',
        ]);

        $data = $request->all();

        if ($request->hasFile('article_file')) {
            $path = $request->file('article_file')->store('service_orders', 'public');
            $data['article_file'] = $path;
        }

        // Set default payment status if not provided (usually pending)
        if (!isset($data['payment_status'])) {
            $data['payment_status'] = 'pending';
        }

        $order = ServiceOrder::create($data);

        // Redirect to a summary/payment page or back with success
        // For now, redirect back with success and the order ID
        return redirect()->route('service.order.summary', ['id' => $order->id])->with('success', 'Order details submitted successfully!');
    }

    public function orderSummary($id)
    {
        $order = ServiceOrder::findOrFail($id);
        return view('pages.service-order-summary', compact('order'));
    }

    // Admin Methods
    public function adminIndex()
    {
        $orders = ServiceOrder::latest()->paginate(20);
        return view('admin.orders.index', compact('orders'));
    }

    public function guestPostOrders()
    {
        $orders = ServiceOrder::where('service_type', 'Guest Post')->latest()->paginate(20);
        $isGuestPostFilter = true;
        return view('admin.orders.index', compact('orders', 'isGuestPostFilter'));
    }

    public function adminShow(ServiceOrder $order)
    {
        return view('admin.orders.show', compact('order'));
    }

    public function adminUpdateStatus(Request $request, ServiceOrder $order)
    {
        $request->validate([
            'order_status' => 'required|in:new,processing,completed,cancelled',
            'payment_status' => 'required|in:pending,completed,failed,refunded',
        ]);

        $order->update($request->only('order_status', 'payment_status'));

        return redirect()->back()->with('success', 'Order updated successfully.');
    }
}
